'use client';

import { useEffect, useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Select } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { LineChart, Line, BarChart, Bar, AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { TrendingUp, Activity, Zap, Brain, AlertCircle } from 'lucide-react';
import StatCard from '@/components/dashboard/stat-card';

interface ForecastData {
  date: string;
  actual: number;
  predicted: number;
  confidence: number;
}

interface CorrelationData {
  metric1: string;
  metric2: string;
  correlation: number;
  strength: string;
}

interface AnomalyData {
  timestamp: string;
  metric: string;
  value: number;
  expected: number;
  deviation: number;
  severity: string;
}

export default function AnalyticsPage() {
  const [forecastData, setForecastData] = useState<ForecastData[]>([]);
  const [correlations, setCorrelations] = useState<CorrelationData[]>([]);
  const [anomalies, setAnomalies] = useState<AnomalyData[]>([]);
  const [period, setPeriod] = useState(30);

  useEffect(() => {
    const loadData = async () => {
      try {
        const response = await fetch(`/api/analytics/advanced?days=${period}`);
        const data = await response.json();
        
        setForecastData(data.forecast || generateMockForecast());
        setCorrelations(data.correlations || generateMockCorrelations());
        setAnomalies(data.anomalies || generateMockAnomalies());
      } catch (error) {
        console.error('Erreur chargement analytics:', error);
        // Données mock en cas d'erreur
        setForecastData(generateMockForecast());
        setCorrelations(generateMockCorrelations());
        setAnomalies(generateMockAnomalies());
      }
    };
    
    loadData();
  }, [period]);

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Analytics Avancés</h1>
          <p className="text-muted-foreground">Intelligence artificielle et prédictions</p>
        </div>
        <div className="flex gap-2">
          <Select value={period.toString()} onValueChange={(v) => setPeriod(Number(v))}>
            <option value="7">7 jours</option>
            <option value="30">30 jours</option>
            <option value="90">90 jours</option>
          </Select>
        </div>
      </div>

      {/* KPIs */}
      <div className="grid gap-4 md:grid-cols-4">
        <StatCard
          title="Précision Prédictions"
          value="94.2%"
          icon={Brain}
          trend={{ value: 2.1, isPositive: true }}
        />
        <StatCard
          title="Anomalies Détectées"
          value={anomalies.length.toString()}
          icon={AlertCircle}
          trend={{ value: -15, isPositive: true }}
        />
        <StatCard
          title="Score Confiance"
          value="0.91"
          icon={Zap}
          trend={{ value: 5.3, isPositive: true }}
        />
        <StatCard
          title="ROI Prédictif"
          value="+23%"
          icon={TrendingUp}
          trend={{ value: 8.7, isPositive: true }}
        />
      </div>

      {/* Contenu principal */}
      <Tabs defaultValue="forecast" className="space-y-4">
        <TabsList>
          <TabsTrigger value="forecast">Prévisions</TabsTrigger>
          <TabsTrigger value="correlations">Corrélations</TabsTrigger>
          <TabsTrigger value="anomalies">Anomalies</TabsTrigger>
          <TabsTrigger value="models">Modèles ML</TabsTrigger>
        </TabsList>

        {/* Prévisions */}
        <TabsContent value="forecast">
          <div className="grid gap-4">
            <Card>
              <CardHeader>
                <CardTitle>Prévisions de Ventes</CardTitle>
                <CardDescription>Prédictions ML sur 7 jours avec intervalles de confiance</CardDescription>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={400}>
                  <LineChart data={forecastData}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="date" />
                    <YAxis />
                    <Tooltip />
                    <Legend />
                    <Line
                      type="monotone"
                      dataKey="actual"
                      stroke="#10b981"
                      strokeWidth={2}
                      name="Réel"
                      dot={{ r: 4 }}
                    />
                    <Line
                      type="monotone"
                      dataKey="predicted"
                      stroke="#3b82f6"
                      strokeWidth={2}
                      strokeDasharray="5 5"
                      name="Prédit"
                      dot={{ r: 4 }}
                    />
                  </LineChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>

            <div className="grid gap-4 md:grid-cols-3">
              <Card>
                <CardHeader>
                  <CardTitle>Prévision J+1</CardTitle>
                  <CardDescription>Ventes demain</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold text-blue-600">
                    {forecastData[forecastData.length - 1]?.predicted.toFixed(0) || 0} €
                  </div>
                  <div className="mt-2 flex items-center gap-2">
                    <div className="text-sm text-muted-foreground">
                      Confiance: {((forecastData[forecastData.length - 1]?.confidence || 0) * 100).toFixed(0)}%
                    </div>
                    <div className="h-2 flex-1 bg-muted rounded-full overflow-hidden">
                      <div
                        className="h-full bg-blue-500"
                        style={{ width: `${(forecastData[forecastData.length - 1]?.confidence || 0) * 100}%` }}
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Prévision Semaine</CardTitle>
                  <CardDescription>CA prévu 7 jours</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold text-green-600">
                    {forecastData.slice(-7).reduce((sum, d) => sum + d.predicted, 0).toFixed(0)} €
                  </div>
                  <div className="mt-2 text-sm text-muted-foreground">
                    Moyenne: {(forecastData.slice(-7).reduce((sum, d) => sum + d.predicted, 0) / 7).toFixed(0)} € /jour
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Tendance Détectée</CardTitle>
                  <CardDescription>Analyse de pattern</CardDescription>
                </CardHeader>
                <CardContent>
                  <Badge className="text-lg px-3 py-1">
                    <TrendingUp className="h-4 w-4 mr-2" />
                    Croissance +12%
                  </Badge>
                  <p className="mt-2 text-sm text-muted-foreground">
                    Tendance haussière confirmée sur 30 jours
                  </p>
                </CardContent>
              </Card>
            </div>
          </div>
        </TabsContent>

        {/* Corrélations */}
        <TabsContent value="correlations">
          <Card>
            <CardHeader>
              <CardTitle>Analyse des Corrélations</CardTitle>
              <CardDescription>Relations entre métriques clés</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {correlations.map((corr, idx) => (
                  <div key={idx} className="p-4 border rounded-lg">
                    <div className="flex items-center justify-between mb-2">
                      <div>
                        <div className="font-medium">{corr.metric1} ↔ {corr.metric2}</div>
                        <div className="text-sm text-muted-foreground">{corr.strength}</div>
                      </div>
                      <Badge variant={Math.abs(corr.correlation) > 0.7 ? 'default' : 'secondary'}>
                        r = {corr.correlation.toFixed(2)}
                      </Badge>
                    </div>
                    <div className="h-2 bg-muted rounded-full overflow-hidden">
                      <div
                        className={`h-full ${corr.correlation > 0 ? 'bg-green-500' : 'bg-red-500'}`}
                        style={{ width: `${Math.abs(corr.correlation) * 100}%` }}
                      />
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          <Card className="mt-4">
            <CardHeader>
              <CardTitle>Facteurs d&apos;Influence</CardTitle>
              <CardDescription>Variables impactant les ventes</CardDescription>
            </CardHeader>
            <CardContent>
              <ResponsiveContainer width="100%" height={300}>
                <BarChart
                  data={[
                    { factor: 'Météo', impact: 85 },
                    { factor: 'Jour Semaine', impact: 72 },
                    { factor: 'Prix Concurrent', impact: 68 },
                    { factor: 'Promotions', impact: 55 },
                    { factor: 'Saison', impact: 48 },
                  ]}
                  layout="vertical"
                >
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis type="number" />
                  <YAxis dataKey="factor" type="category" />
                  <Tooltip />
                  <Bar dataKey="impact" fill="#10b981" name="Impact (%)" />
                </BarChart>
              </ResponsiveContainer>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Anomalies */}
        <TabsContent value="anomalies">
          <Card>
            <CardHeader>
              <CardTitle>Détection d&apos;Anomalies</CardTitle>
              <CardDescription>Alertes ML sur comportements inhabituels</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b">
                      <th className="p-2 text-left">Date/Heure</th>
                      <th className="p-2 text-left">Métrique</th>
                      <th className="p-2 text-right">Valeur</th>
                      <th className="p-2 text-right">Attendu</th>
                      <th className="p-2 text-right">Écart</th>
                      <th className="p-2 text-left">Sévérité</th>
                    </tr>
                  </thead>
                  <tbody>
                    {anomalies.map((anomaly, idx) => (
                      <tr key={idx} className="border-b hover:bg-muted/50">
                        <td className="p-2 text-sm">
                          {new Date(anomaly.timestamp).toLocaleString('fr-FR')}
                        </td>
                        <td className="p-2">
                          <Badge variant="outline">{anomaly.metric}</Badge>
                        </td>
                        <td className="p-2 text-right font-medium">
                          {anomaly.value.toFixed(2)}
                        </td>
                        <td className="p-2 text-right text-muted-foreground">
                          {anomaly.expected.toFixed(2)}
                        </td>
                        <td className="p-2 text-right">
                          <span className={anomaly.deviation > 0 ? 'text-orange-600' : 'text-blue-600'}>
                            {anomaly.deviation > 0 ? '+' : ''}{anomaly.deviation.toFixed(1)}%
                          </span>
                        </td>
                        <td className="p-2">
                          <Badge
                            variant={
                              anomaly.severity === 'HIGH'
                                ? 'destructive'
                                : anomaly.severity === 'MEDIUM'
                                ? 'default'
                                : 'secondary'
                            }
                          >
                            {anomaly.severity}
                          </Badge>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Modèles ML */}
        <TabsContent value="models">
          <div className="grid gap-4 md:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle>Modèles Déployés</CardTitle>
                <CardDescription>État des modèles de prédiction</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  {[
                    { name: 'LSTM Sales Forecast', accuracy: 94.2, status: 'active', lastTrain: '2024-11-08' },
                    { name: 'Random Forest Demand', accuracy: 89.5, status: 'active', lastTrain: '2024-11-07' },
                    { name: 'XGBoost Price Elasticity', accuracy: 91.3, status: 'active', lastTrain: '2024-11-06' },
                    { name: 'Isolation Forest Anomalies', accuracy: 87.8, status: 'training', lastTrain: '2024-11-05' },
                  ].map((model, idx) => (
                    <div key={idx} className="p-4 border rounded-lg space-y-2">
                      <div className="flex items-center justify-between">
                        <div className="font-medium">{model.name}</div>
                        <Badge variant={model.status === 'active' ? 'default' : 'secondary'}>
                          {model.status}
                        </Badge>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-muted-foreground">Précision</span>
                        <span className="font-medium text-green-600">{model.accuracy}%</span>
                      </div>
                      <div className="flex items-center justify-between text-sm">
                        <span className="text-muted-foreground">Dernier entraînement</span>
                        <span>{model.lastTrain}</span>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Performance des Modèles</CardTitle>
                <CardDescription>Métriques d&apos;évaluation</CardDescription>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={300}>
                  <BarChart
                    data={[
                      { metric: 'Accuracy', value: 94.2 },
                      { metric: 'Precision', value: 92.8 },
                      { metric: 'Recall', value: 91.5 },
                      { metric: 'F1-Score', value: 92.1 },
                      { metric: 'RMSE', value: 85.3 },
                    ]}
                  >
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="metric" />
                    <YAxis domain={[0, 100]} />
                    <Tooltip />
                    <Bar dataKey="value" fill="#10b981" />
                  </BarChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>
          </div>

          <Card className="mt-4">
            <CardHeader>
              <CardTitle>Pipeline ML</CardTitle>
              <CardDescription>Architecture des modèles</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                <div className="flex items-center gap-4 p-4 bg-muted rounded-lg">
                  <Activity className="h-8 w-8 text-primary" />
                  <div className="flex-1">
                    <div className="font-medium">Data Ingestion</div>
                    <div className="text-sm text-muted-foreground">MongoDB → Feature Store</div>
                  </div>
                  <Badge>Active</Badge>
                </div>
                <div className="flex items-center gap-4 p-4 bg-muted rounded-lg">
                  <Brain className="h-8 w-8 text-primary" />
                  <div className="flex-1">
                    <div className="font-medium">Feature Engineering</div>
                    <div className="text-sm text-muted-foreground">Transformation & Scaling</div>
                  </div>
                  <Badge>Active</Badge>
                </div>
                <div className="flex items-center gap-4 p-4 bg-muted rounded-lg">
                  <Zap className="h-8 w-8 text-primary" />
                  <div className="flex-1">
                    <div className="font-medium">Model Training</div>
                    <div className="text-sm text-muted-foreground">Auto-ML Pipeline</div>
                  </div>
                  <Badge variant="secondary">Scheduled</Badge>
                </div>
                <div className="flex items-center gap-4 p-4 bg-muted rounded-lg">
                  <TrendingUp className="h-8 w-8 text-primary" />
                  <div className="flex-1">
                    <div className="font-medium">Prediction Service</div>
                    <div className="text-sm text-muted-foreground">Real-time Inference</div>
                  </div>
                  <Badge>Active</Badge>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}

// Fonctions de génération de données mock
function generateMockForecast(): ForecastData[] {
  const data: ForecastData[] = [];
  const baseValue = 2500;
  
  for (let i = -30; i <= 7; i++) {
    const date = new Date();
    date.setDate(date.getDate() + i);
    const actual = i <= 0 ? baseValue + Math.random() * 500 - 250 : 0;
    const predicted = baseValue + Math.sin(i / 7) * 300 + Math.random() * 100;
    
    data.push({
      date: date.toLocaleDateString('fr-FR', { day: '2-digit', month: '2-digit' }),
      actual: actual,
      predicted: predicted,
      confidence: 0.85 + Math.random() * 0.1,
    });
  }
  
  return data;
}

function generateMockCorrelations(): CorrelationData[] {
  return [
    { metric1: 'Température', metric2: 'Ventes Boissons', correlation: 0.87, strength: 'Forte positive' },
    { metric1: 'Prix Diesel', metric2: 'Volume Ventes', correlation: -0.72, strength: 'Forte négative' },
    { metric1: 'Weekend', metric2: 'CA Shop', correlation: 0.65, strength: 'Moyenne positive' },
    { metric1: 'Pluie', metric2: 'Transactions', correlation: -0.45, strength: 'Faible négative' },
    { metric1: 'Promotions', metric2: 'Panier Moyen', correlation: 0.58, strength: 'Moyenne positive' },
  ];
}

function generateMockAnomalies(): AnomalyData[] {
  return [
    {
      timestamp: new Date(Date.now() - 2 * 3600000).toISOString(),
      metric: 'CA Horaire',
      value: 450,
      expected: 280,
      deviation: 60.7,
      severity: 'HIGH',
    },
    {
      timestamp: new Date(Date.now() - 5 * 3600000).toISOString(),
      metric: 'Volume Diesel',
      value: 120,
      expected: 250,
      deviation: -52.0,
      severity: 'MEDIUM',
    },
    {
      timestamp: new Date(Date.now() - 24 * 3600000).toISOString(),
      metric: 'Tx Shop',
      value: 15,
      expected: 45,
      deviation: -66.7,
      severity: 'HIGH',
    },
  ];
}
