import { NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { PurchaseOrder, Product } from '@/models';

/**
 * GET /api/purchases
 * Retourne tous les bons de livraison
 */
export async function GET() {
  try {
    await connectDB();

    const orders = await PurchaseOrder.find()
      .populate({
        path: 'supplierId',
        select: 'supplierName supplierCode',
        options: { strictPopulate: false }
      })
      .sort({ orderDate: -1 })
      .lean();

    return NextResponse.json({
      success: true,
      data: orders,
    });
  } catch (error) {
    console.error('Erreur API Purchases GET:', error);
    return NextResponse.json(
      {
        success: false,
        error: 'Erreur lors de la récupération des bons de livraison',
      },
      { status: 500 }
    );
  }
}

/**
 * POST /api/purchases
 * Crée un nouveau bon de livraison et met à jour les stocks
 */
export async function POST(request: Request) {
  try {
    await connectDB();

    const body = await request.json();
    const { supplierId, orderDate, deliveryDate, items, notes, status } = body;

    // Validation
    if (!supplierId || !orderDate || !items || items.length === 0) {
      return NextResponse.json(
        {
          success: false,
          error: 'Fournisseur, date et articles sont obligatoires',
        },
        { status: 400 }
      );
    }

    // Générer un numéro de bon automatique
    const today = new Date();
    const year = today.getFullYear();
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const count = await PurchaseOrder.countDocuments({
      orderDate: { $gte: new Date(year, today.getMonth(), 1) }
    });
    const orderNumber = `BL${year}${month}${String(count + 1).padStart(4, '0')}`;

    // Calculer le total
    const totalAmount = items.reduce((sum: number, item: { totalPrice: number }) => sum + item.totalPrice, 0);

    // Créer le bon de livraison
    const order = await PurchaseOrder.create({
      orderNumber,
      orderDate: new Date(orderDate),
      supplierId,
      deliveryDate: deliveryDate ? new Date(deliveryDate) : null,
      totalAmount,
      status: status || 'PENDING',
      notes,
      items,
    });

    // Si le statut est RECEIVED, mettre à jour les stocks
    if (status === 'RECEIVED') {
      for (const item of items) {
        const product = await Product.findById(item.productId);
        if (product) {
          product.currentStock = (product.currentStock || 0) + item.quantity;
          await product.save();
        }
      }
    }

    return NextResponse.json({
      success: true,
      message: 'Bon de livraison créé avec succès',
      data: order,
    });
  } catch (error) {
    console.error('Erreur API Purchases POST:', error);
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : 'Erreur lors de la création du bon',
      },
      { status: 500 }
    );
  }
}
