import { NextRequest, NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { PumpTransaction } from '@/models';
import { subDays, format } from 'date-fns';

/**
 * GET /api/sales/daily
 * Ventes quotidiennes par carburant
 */
export async function GET(request: NextRequest) {
  try {
    await connectDB();

    const { searchParams } = new URL(request.url);
    const days = parseInt(searchParams.get('days') || '30');
    
    const endDate = new Date();
    const startDate = subDays(endDate, days);

    const dailySales = await PumpTransaction.aggregate([
      {
        $match: {
          txDatetime: { $gte: startDate, $lte: endDate },
        },
      },
      {
        $group: {
          _id: {
            date: { $dateToString: { format: '%Y-%m-%d', date: '$txDatetime' } },
            fuelType: '$fuelType',
          },
          transactionCount: { $sum: 1 },
          totalLiters: { $sum: '$liters' },
          totalRevenue: { $sum: '$totalAmount' },
          avgLitersPerTx: { $avg: '$liters' },
          avgPrice: { $avg: '$unitPrice' },
          minPrice: { $min: '$unitPrice' },
          maxPrice: { $max: '$unitPrice' },
        },
      },
      {
        $sort: { '_id.date': -1, '_id.fuelType': 1 },
      },
    ]);

    return NextResponse.json({
      success: true,
      data: dailySales.map((item) => ({
        date: item._id.date,
        fuelType: item._id.fuelType,
        transactionCount: item.transactionCount,
        totalLiters: Math.round(item.totalLiters * 100) / 100,
        totalRevenue: Math.round(item.totalRevenue * 100) / 100,
        avgLitersPerTx: Math.round(item.avgLitersPerTx * 100) / 100,
        avgPrice: Math.round(item.avgPrice * 10000) / 10000,
        minPrice: Math.round(item.minPrice * 10000) / 10000,
        maxPrice: Math.round(item.maxPrice * 10000) / 10000,
      })),
    });
  } catch (error) {
    console.error('Erreur API Sales Daily:', error);
    return NextResponse.json(
      { success: false, error: 'Erreur lors de la récupération des ventes' },
      { status: 500 }
    );
  }
}
