import { NextRequest, NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { Tank, TankReading, Delivery, PumpTransaction } from '@/models';
import { startOfDay, endOfDay, subDays, format } from 'date-fns';

interface TankReadingDoc {
  levelLiters: number;
  readingDatetime: Date;
  tankId: unknown;
}

/**
 * GET /api/stocks/reconciliation
 * Réconciliation quotidienne des stocks
 */
export async function GET(request: NextRequest) {
  try {
    await connectDB();

    const { searchParams } = new URL(request.url);
    const days = parseInt(searchParams.get('days') || '7');
    
    const endDate = new Date();
    const startDate = subDays(endDate, days);

    const tanks = await Tank.find({ active: true }).lean();

    const reconciliations = [];

    for (const tank of tanks) {
      // Pour chaque jour
      for (let d = new Date(startDate); d <= endDate; d.setDate(d.getDate() + 1)) {
        const dayStart = startOfDay(d);
        const dayEnd = endOfDay(d);

        // Niveau d'ouverture (première lecture du jour)
        const openingReading = await TankReading.findOne({
          tankId: tank._id,
          readingDatetime: { $gte: dayStart, $lte: dayEnd },
        })
          .sort({ readingDatetime: 1 })
          .lean() as TankReadingDoc | null;

        // Niveau de fermeture (dernière lecture du jour)
        const closingReading = await TankReading.findOne({
          tankId: tank._id,
          readingDatetime: { $gte: dayStart, $lte: dayEnd },
        })
          .sort({ readingDatetime: -1 })
          .lean() as TankReadingDoc | null;

        if (!openingReading || !closingReading) continue;

        // Livraisons du jour
        const deliveries = await Delivery.aggregate([
          {
            $match: {
              tankId: tank._id,
              deliveryDatetime: { $gte: dayStart, $lte: dayEnd },
            },
          },
          {
            $group: {
              _id: null,
              totalDelivered: { $sum: '$volumeDelivered' },
            },
          },
        ]);

        const totalDelivered = deliveries[0]?.totalDelivered || 0;

        // Ventes du jour (pour ce type de carburant)
        const sales = await PumpTransaction.aggregate([
          {
            $match: {
              fuelType: tank.fuelType,
              txDatetime: { $gte: dayStart, $lte: dayEnd },
            },
          },
          {
            $group: {
              _id: null,
              totalSold: { $sum: '$liters' },
            },
          },
        ]);

        const totalSold = sales[0]?.totalSold || 0;

        // Calcul théorique
        const expectedClosing =
          openingReading.levelLiters + totalDelivered - totalSold;
        const variance = closingReading.levelLiters - expectedClosing;
        const variancePct =
          expectedClosing > 0 ? (variance / expectedClosing) * 100 : 0;

        // Classification
        let alertLevel = 'NORMAL';
        if (Math.abs(variance) > 100) alertLevel = 'CRITICAL';
        else if (Math.abs(variance) > 50) alertLevel = 'WARNING';

        reconciliations.push({
          date: format(d, 'yyyy-MM-dd'),
          tankCode: tank.tankCode,
          fuelType: tank.fuelType,
          openingLevel: Math.round(openingReading.levelLiters),
          closingLevel: Math.round(closingReading.levelLiters),
          delivered: Math.round(totalDelivered),
          sold: Math.round(totalSold),
          expectedClosing: Math.round(expectedClosing),
          variance: Math.round(variance),
          variancePct: parseFloat(variancePct.toFixed(2)),
          alertLevel,
        });
      }
    }

    return NextResponse.json({
      success: true,
      data: reconciliations.sort((a, b) => b.date.localeCompare(a.date)),
    });
  } catch (error) {
    console.error('Erreur API Réconciliation:', error);
    return NextResponse.json(
      {
        success: false,
        error: 'Erreur lors de la réconciliation',
      },
      { status: 500 }
    );
  }
}
