import { NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { Tank, TankReading, Delivery, PumpTransaction } from '@/models';
import { startOfDay, endOfDay, subDays } from 'date-fns';

interface TankReadingDoc {
  levelLiters: number;
  readingDatetime: Date;
  tankId: unknown;
  temperature?: number;
}

interface DeliveryDoc {
  deliveryDatetime: Date;
  volumeDelivered: number;
}

/**
 * GET /api/stocks/status
 * Retourne l'état actuel des stocks (cuves)
 */
export async function GET() {
  try {
    await connectDB();

    // Récupérer toutes les cuves actives
    const tanks = await Tank.find({ active: true }).lean();

    const stockStatus = await Promise.all(
      tanks.map(async (tank) => {
        // Dernière lecture
        const latestReading = await TankReading.findOne({ tankId: tank._id })
          .sort({ readingDatetime: -1 })
          .lean() as TankReadingDoc | null;

        // Consommation moyenne (30 derniers jours)
        const thirtyDaysAgo = subDays(new Date(), 30);
        const consumption = await PumpTransaction.aggregate([
          {
            $match: {
              fuelType: tank.fuelType,
              txDatetime: { $gte: thirtyDaysAgo },
            },
          },
          {
            $group: {
              _id: null,
              totalLiters: { $sum: '$liters' },
              daysCount: {
                $addToSet: {
                  $dateToString: { format: '%Y-%m-%d', date: '$txDatetime' },
                },
              },
            },
          },
        ]);

        const avgDailyConsumption =
          consumption.length > 0
            ? consumption[0].totalLiters / consumption[0].daysCount.length
            : 0;

        // Jours de stock restants
        const daysOfStock =
          avgDailyConsumption > 0 && latestReading
            ? latestReading.levelLiters / avgDailyConsumption
            : null;

        // Statut
        let status = 'NORMAL';
        if (latestReading && latestReading.levelLiters < tank.minLevelLiters) {
          status = 'CRITICAL';
        } else if (
          latestReading &&
          latestReading.levelLiters < tank.minLevelLiters * 1.5
        ) {
          status = 'LOW';
        } else if (
          latestReading &&
          latestReading.levelLiters > tank.capacityLiters * 0.85
        ) {
          status = 'HIGH';
        }

        // Dernière livraison
        const lastDelivery = await Delivery.findOne({ tankId: tank._id })
          .sort({ deliveryDatetime: -1 })
          .lean() as DeliveryDoc | null;

        return {
          tank: {
            id: tank._id,
            code: tank.tankCode,
            fuelType: tank.fuelType,
            capacity: tank.capacityLiters,
            minLevel: tank.minLevelLiters,
          },
          currentLevel: latestReading?.levelLiters || 0,
          fillRate:
            latestReading && tank.capacityLiters > 0
              ? (latestReading.levelLiters / tank.capacityLiters) * 100
              : 0,
          avgDailyConsumption,
          daysOfStock,
          status,
          lastReading: latestReading?.readingDatetime || null,
          lastDelivery: lastDelivery?.deliveryDatetime || null,
          temperature: latestReading?.temperature || null,
        };
      })
    );

    return NextResponse.json({
      success: true,
      data: stockStatus,
    });
  } catch (error) {
    console.error('Erreur API Stocks:', error);
    return NextResponse.json(
      {
        success: false,
        error: 'Erreur lors de la récupération des stocks',
      },
      { status: 500 }
    );
  }
}
