'use client';

import { useState, useEffect } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Plus, Trash2, Package, Search, Edit, Filter } from 'lucide-react';

interface Supplier {
  _id: string;
  supplierName: string;
  supplierCode: string;
}

interface Product {
  _id: string;
  productCode: string;
  productName: string;
  purchasePrice: number;
  unit: string;
}

interface PurchaseItem {
  productId: string;
  productCode: string;
  productName: string;
  quantity: number;
  unitPrice: number;
  totalPrice: number;
}

interface PurchaseOrder {
  _id: string;
  orderNumber: string;
  orderDate: string;
  supplierId: { _id: string; supplierName: string } | string | null;
  totalAmount: number;
  status: string;
  items: PurchaseItem[];
  notes?: string;
  deliveryDate?: string;
}

export default function PurchasesPage() {
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [products, setProducts] = useState<Product[]>([]);
  const [orders, setOrders] = useState<PurchaseOrder[]>([]);
  const [loading, setLoading] = useState(true);
  const [message, setMessage] = useState('');
  const [showForm, setShowForm] = useState(false);
  const [editingOrder, setEditingOrder] = useState<PurchaseOrder | null>(null);

  // Formulaire
  const [selectedSupplier, setSelectedSupplier] = useState('');
  const [orderDate, setOrderDate] = useState(new Date().toISOString().split('T')[0]);
  const [deliveryDate, setDeliveryDate] = useState('');
  const [notes, setNotes] = useState('');
  const [items, setItems] = useState<PurchaseItem[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedStatus, setSelectedStatus] = useState<'PENDING' | 'RECEIVED' | 'PARTIAL' | 'CANCELLED'>('PENDING');

  // Filtres de recherche
  const [searchNumber, setSearchNumber] = useState('');
  const [startDate, setStartDate] = useState('');
  const [endDate, setEndDate] = useState('');
  const [filterStatus, setFilterStatus] = useState<string>('ALL');

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    try {
      const [suppliersRes, productsRes, ordersRes] = await Promise.all([
        fetch('/api/suppliers'),
        fetch('/api/products'),
        fetch('/api/purchases'),
      ]);

      const suppliersData = await suppliersRes.json();
      const productsData = await productsRes.json();
      const ordersData = await ordersRes.json();

      if (suppliersData.success) setSuppliers(suppliersData.data);
      if (productsData.success) setProducts(productsData.data);
      if (ordersData.success) setOrders(ordersData.data);
    } catch (error) {
      console.error('Erreur chargement:', error);
      setMessage('❌ Erreur de chargement');
    } finally {
      setLoading(false);
    }
  };

  const addItem = (product: Product) => {
    const existingIndex = items.findIndex(item => item.productId === product._id);
    
    if (existingIndex >= 0) {
      // Incrémenter la quantité si déjà présent
      const newItems = [...items];
      newItems[existingIndex].quantity += 1;
      newItems[existingIndex].totalPrice = newItems[existingIndex].quantity * newItems[existingIndex].unitPrice;
      setItems(newItems);
    } else {
      // Ajouter nouveau
      setItems([...items, {
        productId: product._id,
        productCode: product.productCode,
        productName: product.productName,
        quantity: 1,
        unitPrice: product.purchasePrice || 0,
        totalPrice: product.purchasePrice || 0,
      }]);
    }
    setSearchTerm('');
  };

  const updateItem = (index: number, field: 'quantity' | 'unitPrice', value: number) => {
    const newItems = [...items];
    newItems[index][field] = value;
    newItems[index].totalPrice = newItems[index].quantity * newItems[index].unitPrice;
    setItems(newItems);
  };

  const removeItem = (index: number) => {
    setItems(items.filter((_, i) => i !== index));
  };

  const calculateTotal = () => {
    return items.reduce((sum, item) => sum + item.totalPrice, 0);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!selectedSupplier || items.length === 0) {
      setMessage('❌ Veuillez sélectionner un fournisseur et ajouter des articles');
      return;
    }

    try {
      const url = editingOrder ? `/api/purchases/${editingOrder._id}` : '/api/purchases';
      const method = editingOrder ? 'PUT' : 'POST';

      const response = await fetch(url, {
        method,
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          supplierId: selectedSupplier,
          orderDate,
          deliveryDate: deliveryDate || null,
          notes,
          items,
          status: selectedStatus,
        }),
      });

      const data = await response.json();

      if (data.success) {
        setMessage(editingOrder ? '✅ Bon modifié avec succès' : '✅ Bon de livraison créé avec succès');
        resetForm();
        loadData();
        setShowForm(false);
        setEditingOrder(null);
      } else {
        setMessage(`❌ ${data.error}`);
      }
    } catch (error) {
      console.error('Erreur création:', error);
      setMessage('❌ Erreur lors de la création');
    }
  };

  const resetForm = () => {
    setSelectedSupplier('');
    setOrderDate(new Date().toISOString().split('T')[0]);
    setDeliveryDate('');
    setNotes('');
    setItems([]);
    setSearchTerm('');
    setEditingOrder(null);
    setSelectedStatus('PENDING');
  };

  const deleteOrder = async (id: string) => {
    if (!confirm('Annuler ce bon de livraison ?')) return;

    try {
      const response = await fetch(`/api/purchases/${id}`, {
        method: 'DELETE',
      });

      const data = await response.json();

      if (data.success) {
        setMessage('✅ Bon annulé');
        loadData();
      } else {
        setMessage(`❌ ${data.error}`);
      }
    } catch (error) {
      console.error('Erreur suppression:', error);
      setMessage('❌ Erreur');
    }
  };

  const editOrder = (order: PurchaseOrder) => {
    setEditingOrder(order);
    
    // Gérer le cas où supplierId est null ou undefined
    const supplierId = order.supplierId && typeof order.supplierId === 'object' 
      ? order.supplierId._id 
      : order.supplierId || '';
    
    setSelectedSupplier(supplierId);
    setOrderDate(order.orderDate.split('T')[0]);
    setDeliveryDate('');
    setNotes(order.notes || '');
    setItems([...order.items]);
    setSelectedStatus(order.status as 'PENDING' | 'RECEIVED' | 'PARTIAL' | 'CANCELLED');
    setShowForm(true);
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  const filteredProducts = products.filter(p =>
    p.productName.toLowerCase().includes(searchTerm.toLowerCase()) ||
    p.productCode.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const filteredOrders = orders.filter(order => {
    // Filtre par numéro
    if (searchNumber && !order.orderNumber.toLowerCase().includes(searchNumber.toLowerCase())) {
      return false;
    }

    // Filtre par statut
    if (filterStatus !== 'ALL' && order.status !== filterStatus) {
      return false;
    }

    // Filtre par date de début
    if (startDate && new Date(order.orderDate) < new Date(startDate)) {
      return false;
    }

    // Filtre par date de fin
    if (endDate && new Date(order.orderDate) > new Date(endDate)) {
      return false;
    }

    return true;
  });

  if (loading) {
    return <div className="p-8">Chargement...</div>;
  }

  return (
    <div className="p-8">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Bons de Livraison</h1>
        <Button onClick={() => {
          if (showForm && editingOrder) {
            // Si on est en mode édition, réinitialiser
            resetForm();
          }
          setShowForm(!showForm);
        }}>
          <Plus className="mr-2 h-4 w-4" />
          Nouveau Bon
        </Button>
      </div>

      {message && (
        <div className="mb-4 p-3 bg-blue-50 border border-blue-200 rounded">
          {message}
        </div>
      )}

      {/* Filtres de recherche */}
      <Card className="mb-6">
        <CardHeader className="pb-3">
          <CardTitle className="text-lg flex items-center gap-2">
            <Filter className="h-5 w-5" />
            Filtres de recherche
          </CardTitle>
        </CardHeader>
        <CardContent className="p-4 pt-0">
          <div className="grid grid-cols-4 gap-4">
            <div>
              <label className="block text-sm font-medium mb-1">Numéro de bon</label>
              <input
                type="text"
                value={searchNumber}
                onChange={(e) => setSearchNumber(e.target.value)}
                placeholder="BL2025110003"
                className="w-full border rounded p-2"
              />
            </div>
            <div>
              <label className="block text-sm font-medium mb-1">Date début</label>
              <input
                type="date"
                value={startDate}
                onChange={(e) => setStartDate(e.target.value)}
                className="w-full border rounded p-2"
              />
            </div>
            <div>
              <label className="block text-sm font-medium mb-1">Date fin</label>
              <input
                type="date"
                value={endDate}
                onChange={(e) => setEndDate(e.target.value)}
                className="w-full border rounded p-2"
              />
            </div>
            <div>
              <label className="block text-sm font-medium mb-1">Statut</label>
              <select
                value={filterStatus}
                onChange={(e) => setFilterStatus(e.target.value)}
                className="w-full border rounded p-2"
              >
                <option value="ALL">Tous</option>
                <option value="PENDING">En attente</option>
                <option value="RECEIVED">Reçu</option>
                <option value="PARTIAL">Partiel</option>
                <option value="CANCELLED">Annulé</option>
              </select>
            </div>
          </div>
          <div className="mt-3 flex justify-between items-center text-sm text-gray-600">
            <span>{filteredOrders.length} bon(s) trouvé(s)</span>
            {(searchNumber || startDate || endDate || filterStatus !== 'ALL') && (
              <Button
                variant="outline"
                size="sm"
                onClick={() => {
                  setSearchNumber('');
                  setStartDate('');
                  setEndDate('');
                  setFilterStatus('ALL');
                }}
              >
                Réinitialiser les filtres
              </Button>
            )}
          </div>
        </CardContent>
      </Card>

      {showForm && (
        <Card className="mb-6">
          <CardHeader>
            <CardTitle>
              {editingOrder ? `Modifier le Bon ${editingOrder.orderNumber}` : 'Créer un Bon de Livraison'}
            </CardTitle>
            {editingOrder && editingOrder.status === 'RECEIVED' && (
              <p className="text-sm text-orange-600 mt-2">
                ⚠️ Attention : Ce bon a déjà été reçu. Les modifications impacteront les stocks.
              </p>
            )}
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              {/* Sélection fournisseur */}
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Fournisseur *</label>
                  <select
                    value={selectedSupplier}
                    onChange={(e) => setSelectedSupplier(e.target.value)}
                    className="w-full border rounded p-2"
                    required
                  >
                    <option value="">Sélectionner...</option>
                    {suppliers.map(s => (
                      <option key={s._id} value={s._id}>
                        {s.supplierName} ({s.supplierCode})
                      </option>
                    ))}
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium mb-1">Date de livraison *</label>
                  <input
                    type="date"
                    value={orderDate}
                    onChange={(e) => setOrderDate(e.target.value)}
                    className="w-full border rounded p-2"
                    required
                  />
                </div>
              </div>

              {/* Statut */}
              <div>
                <label className="block text-sm font-medium mb-1">Statut *</label>
                <select
                  value={selectedStatus}
                  onChange={(e) => setSelectedStatus(e.target.value as 'PENDING' | 'RECEIVED' | 'PARTIAL' | 'CANCELLED')}
                  className="w-full border rounded p-2"
                  required
                >
                  <option value="PENDING">En attente</option>
                  <option value="RECEIVED">Reçu (stock mis à jour)</option>
                  <option value="PARTIAL">Partiel</option>
                  <option value="CANCELLED">Annulé</option>
                </select>
                {selectedStatus === 'RECEIVED' && (
                  <p className="text-sm text-green-600 mt-1">
                    ✅ Les stocks seront mis à jour automatiquement
                  </p>
                )}
                {selectedStatus === 'PENDING' && (
                  <p className="text-sm text-orange-600 mt-1">
                    ⏳ Les stocks ne seront pas encore mis à jour
                  </p>
                )}
              </div>

              {/* Recherche produit */}
              <div>
                <label className="block text-sm font-medium mb-1">Rechercher un article</label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <input
                    type="text"
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    placeholder="Code ou nom de l'article..."
                    className="w-full border rounded p-2 pl-10"
                  />
                </div>

                {searchTerm && filteredProducts.length > 0 && (
                  <div className="mt-2 max-h-48 overflow-y-auto border rounded bg-white shadow-lg">
                    {filteredProducts.slice(0, 10).map(product => (
                      <div
                        key={product._id}
                        onClick={() => addItem(product)}
                        className="p-2 hover:bg-gray-100 cursor-pointer border-b last:border-b-0"
                      >
                        <div className="font-medium">{product.productName}</div>
                        <div className="text-sm text-gray-500">
                          {product.productCode} - {product.purchasePrice?.toFixed(2)}€
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </div>

              {/* Liste des articles */}
              {items.length > 0 && (
                <div className="border rounded p-4">
                  <h3 className="font-medium mb-3">Articles ({items.length})</h3>
                  <div className="space-y-2">
                    {items.map((item, index) => (
                      <div key={index} className="grid grid-cols-12 gap-2 items-center bg-gray-50 p-2 rounded">
                        <div className="col-span-4">
                          <div className="font-medium text-sm">{item.productName}</div>
                          <div className="text-xs text-gray-500">{item.productCode}</div>
                        </div>
                        <div className="col-span-2">
                          <input
                            type="number"
                            value={item.quantity}
                            onChange={(e) => updateItem(index, 'quantity', parseFloat(e.target.value) || 0)}
                            min="1"
                            step="1"
                            className="w-full border rounded p-1 text-sm"
                          />
                        </div>
                        <div className="col-span-2">
                          <input
                            type="number"
                            value={item.unitPrice}
                            onChange={(e) => updateItem(index, 'unitPrice', parseFloat(e.target.value) || 0)}
                            min="0"
                            step="0.01"
                            className="w-full border rounded p-1 text-sm"
                          />
                        </div>
                        <div className="col-span-2 text-right font-medium">
                          {item.totalPrice.toFixed(2)}€
                        </div>
                        <div className="col-span-2 text-right">
                          <Button
                            type="button"
                            onClick={() => removeItem(index)}
                            variant="ghost"
                            size="sm"
                          >
                            <Trash2 className="h-4 w-4 text-red-500" />
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>

                  <div className="mt-4 pt-4 border-t flex justify-end">
                    <div className="text-right">
                      <div className="text-sm text-gray-500">Total</div>
                      <div className="text-2xl font-bold">{calculateTotal().toFixed(2)}€</div>
                    </div>
                  </div>
                </div>
              )}

              {/* Notes */}
              <div>
                <label className="block text-sm font-medium mb-1">Notes</label>
                <textarea
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  rows={3}
                  className="w-full border rounded p-2"
                  placeholder="Remarques..."
                />
              </div>

              {/* Boutons */}
              <div className="flex gap-2">
                <Button type="submit" disabled={items.length === 0}>
                  <Package className="mr-2 h-4 w-4" />
                  {editingOrder ? 'Modifier le Bon' : 'Enregistrer le Bon'}
                </Button>
                <Button type="button" variant="outline" onClick={() => {
                  resetForm();
                  setShowForm(false);
                }}>
                  Annuler
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      )}

      {/* Liste des bons */}
      <div className="grid gap-4">
        {filteredOrders.map(order => (
          <Card key={order._id}>
            <CardContent className="p-4">
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="text-lg font-bold">{order.orderNumber}</h3>
                    <span className={`px-2 py-1 rounded text-xs font-medium ${
                      order.status === 'RECEIVED' ? 'bg-green-100 text-green-800' :
                      order.status === 'CANCELLED' ? 'bg-red-100 text-red-800' :
                      'bg-yellow-100 text-yellow-800'
                    }`}>
                      {order.status}
                    </span>
                  </div>
                  
                  <div className="grid grid-cols-3 gap-4 text-sm mb-3">
                    <div>
                      <span className="text-gray-500">Fournisseur:</span>
                      <div className="font-medium">
                        {order.supplierId && typeof order.supplierId === 'object' 
                          ? order.supplierId.supplierName 
                          : 'Non spécifié'}
                      </div>
                    </div>
                    <div>
                      <span className="text-gray-500">Date:</span>
                      <div className="font-medium">
                        {new Date(order.orderDate).toLocaleDateString('fr-FR')}
                      </div>
                    </div>
                    <div>
                      <span className="text-gray-500">Montant:</span>
                      <div className="font-bold text-lg">{order.totalAmount.toFixed(2)}€</div>
                    </div>
                  </div>

                  <div className="bg-gray-50 rounded p-3">
                    <div className="text-sm font-medium mb-2">Articles ({order.items.length})</div>
                    <div className="space-y-1">
                      {order.items.slice(0, 3).map((item, idx) => (
                        <div key={idx} className="text-sm flex justify-between">
                          <span>{item.productName} ({item.quantity}x)</span>
                          <span className="font-medium">{item.totalPrice.toFixed(2)}€</span>
                        </div>
                      ))}
                      {order.items.length > 3 && (
                        <div className="text-xs text-gray-500">
                          ... et {order.items.length - 3} autre(s)
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                <div className="ml-4 flex gap-2">
                  {order.status !== 'CANCELLED' && (
                    <>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => editOrder(order)}
                        title="Modifier"
                      >
                        <Edit className="h-4 w-4 text-blue-500" />
                      </Button>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => deleteOrder(order._id)}
                        title="Annuler"
                      >
                        <Trash2 className="h-4 w-4 text-red-500" />
                      </Button>
                    </>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>
        ))}

        {filteredOrders.length === 0 && (
          <Card>
            <CardContent className="p-8 text-center text-gray-500">
              {orders.length === 0 
                ? 'Aucun bon de livraison' 
                : 'Aucun bon ne correspond aux critères de recherche'}
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
}
