/* eslint-disable @typescript-eslint/no-explicit-any */
'use client';

import { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Select } from '@/components/ui/select';
import { Settings, Database, Bell, Users, Shield, Palette, Globe, Droplet, Fuel, Plus, Trash2, Edit } from 'lucide-react';

// Composant de gestion des citernes et pompes
function TanksAndPumpsManagement() {
  const [tanks, setTanks] = useState<any[]>([]);
  const [pumps, setPumps] = useState([
    { pump_id: 1, pumpCode: 'P01', fuels: ['DIESEL', 'SP95'], tank_mapping: { DIESEL: 1, SP95: 2 } },
    { pump_id: 2, pumpCode: 'P02', fuels: ['DIESEL', 'SP95'], tank_mapping: { DIESEL: 1, SP95: 2 } },
    { pump_id: 3, pumpCode: 'P03', fuels: ['SP95', 'SP98'], tank_mapping: { SP95: 2, SP98: 3 } },
    { pump_id: 4, pumpCode: 'P04', fuels: ['SP95', 'SP98'], tank_mapping: { SP95: 2, SP98: 3 } },
    { pump_id: 5, pumpCode: 'P05', fuels: ['DIESEL'], tank_mapping: { DIESEL: 1 } },
    { pump_id: 6, pumpCode: 'P06', fuels: ['DIESEL', 'E85'], tank_mapping: { DIESEL: 1, E85: 4 } },
  ]);

  const [editingLevel, setEditingLevel] = useState<{tankId: string, currentLevel: number, newLevel: number} | null>(null);
  const [submitting, setSubmitting] = useState(false);
  const [message, setMessage] = useState<{type: 'success' | 'error', text: string} | null>(null);

  const fuelTypes = ['DIESEL', 'SP95', 'SP98', 'E85'];
  const fuelColors = {
    DIESEL: 'bg-yellow-100 text-yellow-800 border-yellow-300',
    SP95: 'bg-green-100 text-green-800 border-green-300',
    SP98: 'bg-blue-100 text-blue-800 border-blue-300',
    E85: 'bg-emerald-100 text-emerald-800 border-emerald-300',
  };

  // Charger les citernes depuis l'API
  useEffect(() => {
    fetchTanks();
  }, []);

  const fetchTanks = async () => {
    try {
      const response = await fetch('/api/tanks');
      const data = await response.json();
      console.log('📊 Citernes chargées:', data.data);
      if (data.success) {
        setTanks(data.data);
      }
    } catch (error) {
      console.error('Erreur chargement citernes:', error);
    }
  };

  const handleUpdateLevel = async () => {
    if (!editingLevel) return;

    setSubmitting(true);
    setMessage(null);

    try {
      const response = await fetch('/api/tanks/update-level', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          tankId: editingLevel.tankId,
          newLevel: editingLevel.newLevel,
          readingType: 'MANUAL'
        })
      });

      const data = await response.json();

      if (data.success) {
        setMessage({ type: 'success', text: 'Niveau mis à jour avec succès!' });
        setEditingLevel(null);
        fetchTanks(); // Recharger les citernes
      } else {
        setMessage({ type: 'error', text: data.error || 'Erreur lors de la mise à jour' });
      }
    } catch (error) {
      setMessage({ type: 'error', text: 'Erreur de connexion au serveur' });
    } finally {
      setSubmitting(false);
    }
  };

  const handleAddTank = () => {
    alert('Fonctionnalité en développement');
  };

  const handleDeleteTank = (tankId: string) => {
    if (confirm('Êtes-vous sûr de vouloir supprimer cette citerne ?')) {
      alert('Fonctionnalité en développement');
    }
  };

  const togglePumpConnection = (tankId: number, pumpId: number) => {
    alert('Fonctionnalité en développement');
  };

  return (
    <div className="grid gap-6">
      {/* Liste des citernes */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Droplet className="h-5 w-5" />
            Gestion des Citernes
          </CardTitle>
          <CardDescription>Configurez vos citernes et leurs pompes connectées</CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Citernes existantes */}
          <div className="space-y-4">
            {tanks.map((tank) => (
              <div key={`${tank._id}-${tank.currentLevel}`} className="border rounded-lg p-4 space-y-4">
                <div className="flex items-start justify-between">
                  <div className="space-y-1 flex-1">
                    <div className="flex items-center gap-2">\
                      <h3 className="font-semibold text-lg">{tank.tankCode}</h3>
                      <Badge className={fuelColors[tank.fuelType as keyof typeof fuelColors] || fuelColors.DIESEL}>{tank.fuelType}</Badge>
                    </div>
                    <div className="text-sm text-muted-foreground">
                      Capacité: {tank.capacityLiters.toLocaleString()} L | 
                      Minimum: {tank.minLevelLiters.toLocaleString()} L
                    </div>
                    <div className="mt-2">
                      <div className="flex items-center gap-4">
                        <div>
                          <span className="text-sm font-medium">Stock actuel:</span>
                          <span className={`ml-2 text-lg font-bold ${tank.currentLevel < tank.minLevelLiters ? 'text-red-600' : 'text-green-600'}`}>
                            {tank.currentLevel.toLocaleString()} L
                          </span>
                          <span className="text-sm text-muted-foreground ml-2">
                            ({tank.fillPercentage.toFixed(1)}%)
                          </span>
                        </div>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => setEditingLevel({
                            tankId: tank._id,
                            currentLevel: tank.currentLevel,
                            newLevel: tank.currentLevel
                          })}
                        >
                          <Edit className="h-4 w-4 mr-1" />
                          Modifier Stock
                        </Button>
                      </div>
                      
                      {/* Barre de progression */}
                      <div className="mt-2 w-full bg-gray-200 rounded-full h-3 overflow-hidden">
                        <div 
                          className={`h-full transition-all ${
                            tank.fillPercentage > 50 ? 'bg-green-500' :
                            tank.fillPercentage > 25 ? 'bg-yellow-500' : 'bg-red-500'
                          }`}
                          style={{ width: `${Math.min(tank.fillPercentage, 100)}%` }}
                        />
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>

          {/* Modal de modification du stock */}
          {editingLevel && (
            <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
              <Card className="w-full max-w-md">
                <CardHeader className="bg-blue-50">
                  <CardTitle>Modifier le Stock de Citerne</CardTitle>
                  <CardDescription>
                    Niveau actuel: {editingLevel.currentLevel.toLocaleString()} L
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-4 pt-6">
                  <div>
                    <Label htmlFor="newLevel">Nouveau Niveau (Litres) *</Label>
                    <Input
                      id="newLevel"
                      type="number"
                      step="0.01"
                      min="0"
                      value={editingLevel.newLevel}
                      onChange={(e) => setEditingLevel({
                        ...editingLevel,
                        newLevel: parseFloat(e.target.value) || 0
                      })}
                      className="text-lg font-semibold"
                    />
                    <p className="text-sm text-muted-foreground mt-1">
                      Différence: {(editingLevel.newLevel - editingLevel.currentLevel).toLocaleString()} L
                    </p>
                  </div>

                  {message && (
                    <div className={`p-3 rounded ${
                      message.type === 'success' ? 'bg-green-50 text-green-800' : 'bg-red-50 text-red-800'
                    }`}>
                      {message.text}
                    </div>
                  )}

                  <div className="flex gap-2">
                    <Button
                      variant="outline"
                      onClick={() => {
                        setEditingLevel(null);
                        setMessage(null);
                      }}
                      disabled={submitting}
                    >
                      Annuler
                    </Button>
                    <Button
                      onClick={handleUpdateLevel}
                      disabled={submitting}
                      className="flex-1"
                    >
                      {submitting ? 'Enregistrement...' : 'Enregistrer'}
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          )}

        </CardContent>
      </Card>
    </div>
  );
}

export default function SettingsPage() {
  const [settings, setSettings] = useState({
    // Général
    stationName: 'Station Service Analytics',
    stationAddress: '123 Route Nationale, 75000 Paris',
    currency: 'EUR',
    language: 'fr-FR',
    timezone: 'Europe/Paris',
    
    // Alertes
    emailNotifications: true,
    smsNotifications: false,
    criticalAlertEmail: 'admin@station.com',
    lowStockThreshold: 15,
    criticalStockThreshold: 5,
    
    // Base de données
    mongoUri: 'mongodb://localhost:27017/station-service',
    backupEnabled: true,
    backupFrequency: 'daily',
    
    // API
    apiTimeout: 30,
    apiRateLimit: 100,
    
    // Sécurité
    sessionTimeout: 30,
    passwordPolicy: 'strong',
    twoFactorAuth: false,
  });

  const handleSave = () => {
    console.log('Saving settings:', settings);
    alert('Paramètres sauvegardés avec succès !');
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Paramètres</h1>
          <p className="text-muted-foreground">Configuration de l&apos;application</p>
        </div>
        <Button onClick={handleSave}>
          <Settings className="h-4 w-4 mr-2" />
          Sauvegarder
        </Button>
      </div>

      {/* Contenu */}
      <Tabs defaultValue="general" className="space-y-4">
        <TabsList>
          <TabsTrigger value="general">
            <Globe className="h-4 w-4 mr-2" />
            Général
          </TabsTrigger>
          <TabsTrigger value="tanks">
            <Droplet className="h-4 w-4 mr-2" />
            Citernes & Pompes
          </TabsTrigger>
          <TabsTrigger value="alerts">
            <Bell className="h-4 w-4 mr-2" />
            Alertes
          </TabsTrigger>
          <TabsTrigger value="database">
            <Database className="h-4 w-4 mr-2" />
            Base de Données
          </TabsTrigger>
          <TabsTrigger value="users">
            <Users className="h-4 w-4 mr-2" />
            Utilisateurs
          </TabsTrigger>
          <TabsTrigger value="security">
            <Shield className="h-4 w-4 mr-2" />
            Sécurité
          </TabsTrigger>
          <TabsTrigger value="appearance">
            <Palette className="h-4 w-4 mr-2" />
            Apparence
          </TabsTrigger>
        </TabsList>

        {/* Général */}
        <TabsContent value="general">
          <Card>
            <CardHeader>
              <CardTitle>Informations Générales</CardTitle>
              <CardDescription>Configuration de base de la station</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="stationName">Nom de la Station</Label>
                <Input
                  id="stationName"
                  value={settings.stationName}
                  onChange={(e) => setSettings({ ...settings, stationName: e.target.value })}
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="stationAddress">Adresse</Label>
                <Input
                  id="stationAddress"
                  value={settings.stationAddress}
                  onChange={(e) => setSettings({ ...settings, stationAddress: e.target.value })}
                />
              </div>

              <div className="grid gap-4 md:grid-cols-3">
                <div className="space-y-2">
                  <Label htmlFor="currency">Devise</Label>
                  <Select
                    value={settings.currency}
                    onValueChange={(value) => setSettings({ ...settings, currency: value })}
                  >
                    <option value="EUR">Euro (€)</option>
                    <option value="USD">Dollar ($)</option>
                    <option value="GBP">Livre (£)</option>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="language">Langue</Label>
                  <Select
                    value={settings.language}
                    onValueChange={(value) => setSettings({ ...settings, language: value })}
                  >
                    <option value="fr-FR">Français</option>
                    <option value="en-US">English</option>
                    <option value="es-ES">Español</option>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="timezone">Fuseau Horaire</Label>
                  <Select
                    value={settings.timezone}
                    onValueChange={(value) => setSettings({ ...settings, timezone: value })}
                  >
                    <option value="Europe/Paris">Europe/Paris (GMT+1)</option>
                    <option value="America/New_York">America/New_York (GMT-5)</option>
                    <option value="Asia/Tokyo">Asia/Tokyo (GMT+9)</option>
                  </Select>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Citernes & Pompes */}
        <TabsContent value="tanks">
          <TanksAndPumpsManagement />
        </TabsContent>

        {/* Alertes */}
        <TabsContent value="alerts">
          <div className="grid gap-4">
            <Card>
              <CardHeader>
                <CardTitle>Notifications</CardTitle>
                <CardDescription>Configuration des alertes et notifications</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label htmlFor="emailNotifications">Notifications Email</Label>
                    <p className="text-sm text-muted-foreground">Recevoir les alertes par email</p>
                  </div>
                  <input
                    type="checkbox"
                    id="emailNotifications"
                    checked={settings.emailNotifications}
                    onChange={(e) => setSettings({ ...settings, emailNotifications: e.target.checked })}
                    className="h-4 w-4"
                  />
                </div>

                <div className="flex items-center justify-between">
                  <div>
                    <Label htmlFor="smsNotifications">Notifications SMS</Label>
                    <p className="text-sm text-muted-foreground">Recevoir les alertes critiques par SMS</p>
                  </div>
                  <input
                    type="checkbox"
                    id="smsNotifications"
                    checked={settings.smsNotifications}
                    onChange={(e) => setSettings({ ...settings, smsNotifications: e.target.checked })}
                    className="h-4 w-4"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="criticalAlertEmail">Email Alertes Critiques</Label>
                  <Input
                    id="criticalAlertEmail"
                    type="email"
                    value={settings.criticalAlertEmail}
                    onChange={(e) => setSettings({ ...settings, criticalAlertEmail: e.target.value })}
                  />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Seuils d&apos;Alerte</CardTitle>
                <CardDescription>Définir les seuils de déclenchement</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-2">
                    <Label htmlFor="lowStockThreshold">Seuil Stock Bas (%)</Label>
                    <Input
                      id="lowStockThreshold"
                      type="number"
                      value={settings.lowStockThreshold}
                      onChange={(e) => setSettings({ ...settings, lowStockThreshold: Number(e.target.value) })}
                    />
                    <p className="text-xs text-muted-foreground">Alerte quand le niveau descend sous ce seuil</p>
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="criticalStockThreshold">Seuil Stock Critique (%)</Label>
                    <Input
                      id="criticalStockThreshold"
                      type="number"
                      value={settings.criticalStockThreshold}
                      onChange={(e) => setSettings({ ...settings, criticalStockThreshold: Number(e.target.value) })}
                    />
                    <p className="text-xs text-muted-foreground">Alerte critique urgente</p>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        {/* Base de données */}
        <TabsContent value="database">
          <div className="grid gap-4">
            <Card>
              <CardHeader>
                <CardTitle>Connexion MongoDB</CardTitle>
                <CardDescription>Configuration de la base de données</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="mongoUri">URI MongoDB</Label>
                  <Input
                    id="mongoUri"
                    value={settings.mongoUri}
                    onChange={(e) => setSettings({ ...settings, mongoUri: e.target.value })}
                    placeholder="mongodb://localhost:27017/station-service"
                  />
                  <p className="text-xs text-muted-foreground">
                    Format: mongodb://[username:password@]host:port/database
                  </p>
                </div>

                <div className="p-4 bg-muted rounded-lg">
                  <div className="flex items-center justify-between">
                    <div>
                      <div className="font-medium">Statut Connexion</div>
                      <div className="text-sm text-muted-foreground">MongoDB Server</div>
                    </div>
                    <Badge>Connecté</Badge>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Sauvegardes</CardTitle>
                <CardDescription>Gestion des backups automatiques</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex items-center justify-between">
                  <div>
                    <Label htmlFor="backupEnabled">Sauvegardes Automatiques</Label>
                    <p className="text-sm text-muted-foreground">Activer les backups quotidiens</p>
                  </div>
                  <input
                    type="checkbox"
                    id="backupEnabled"
                    checked={settings.backupEnabled}
                    onChange={(e) => setSettings({ ...settings, backupEnabled: e.target.checked })}
                    className="h-4 w-4"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="backupFrequency">Fréquence</Label>
                  <Select
                    value={settings.backupFrequency}
                    onValueChange={(value) => setSettings({ ...settings, backupFrequency: value })}
                  >
                    <option value="hourly">Toutes les heures</option>
                    <option value="daily">Quotidienne</option>
                    <option value="weekly">Hebdomadaire</option>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Button variant="outline" className="w-full">
                    <Database className="h-4 w-4 mr-2" />
                    Lancer Sauvegarde Manuelle
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        {/* Utilisateurs */}
        <TabsContent value="users">
          <Card>
            <CardHeader>
              <CardTitle>Gestion Utilisateurs</CardTitle>
              <CardDescription>Comptes et permissions</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {[
                  { name: 'Admin Principal', email: 'admin@station.com', role: 'Admin', status: 'Active' },
                  { name: 'Manager', email: 'manager@station.com', role: 'Manager', status: 'Active' },
                  { name: 'Opérateur', email: 'operator@station.com', role: 'Operator', status: 'Active' },
                ].map((user, idx) => (
                  <div key={idx} className="flex items-center justify-between p-4 border rounded-lg">
                    <div>
                      <div className="font-medium">{user.name}</div>
                      <div className="text-sm text-muted-foreground">{user.email}</div>
                    </div>
                    <div className="flex items-center gap-3">
                      <Badge>{user.role}</Badge>
                      <Badge variant="outline">{user.status}</Badge>
                      <Button variant="ghost" size="sm">Modifier</Button>
                    </div>
                  </div>
                ))}
              </div>

              <Button className="w-full mt-4">
                <Users className="h-4 w-4 mr-2" />
                Ajouter Utilisateur
              </Button>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Sécurité */}
        <TabsContent value="security">
          <div className="grid gap-4">
            <Card>
              <CardHeader>
                <CardTitle>Paramètres de Sécurité</CardTitle>
                <CardDescription>Configuration de l&apos;authentification</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="sessionTimeout">Timeout Session (minutes)</Label>
                  <Input
                    id="sessionTimeout"
                    type="number"
                    value={settings.sessionTimeout}
                    onChange={(e) => setSettings({ ...settings, sessionTimeout: Number(e.target.value) })}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="passwordPolicy">Politique Mot de Passe</Label>
                  <Select
                    value={settings.passwordPolicy}
                    onValueChange={(value) => setSettings({ ...settings, passwordPolicy: value })}
                  >
                    <option value="basic">Basique (min 6 caractères)</option>
                    <option value="medium">Moyen (8+ avec chiffres)</option>
                    <option value="strong">Fort (12+ avec symboles)</option>
                  </Select>
                </div>

                <div className="flex items-center justify-between">
                  <div>
                    <Label htmlFor="twoFactorAuth">Authentification 2 Facteurs</Label>
                    <p className="text-sm text-muted-foreground">Sécurité renforcée avec code OTP</p>
                  </div>
                  <input
                    type="checkbox"
                    id="twoFactorAuth"
                    checked={settings.twoFactorAuth}
                    onChange={(e) => setSettings({ ...settings, twoFactorAuth: e.target.checked })}
                    className="h-4 w-4"
                  />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Logs d&apos;Audit</CardTitle>
                <CardDescription>Dernières activités système</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  {[
                    { action: 'Connexion utilisateur', user: 'admin@station.com', time: '2024-11-10 14:35' },
                    { action: 'Modification paramètres', user: 'manager@station.com', time: '2024-11-10 12:20' },
                    { action: 'Export données', user: 'admin@station.com', time: '2024-11-10 09:15' },
                  ].map((log, idx) => (
                    <div key={idx} className="flex items-center justify-between p-3 bg-muted rounded text-sm">
                      <div>
                        <div className="font-medium">{log.action}</div>
                        <div className="text-muted-foreground">{log.user}</div>
                      </div>
                      <div className="text-muted-foreground">{log.time}</div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        {/* Apparence */}
        <TabsContent value="appearance">
          <Card>
            <CardHeader>
              <CardTitle>Personnalisation</CardTitle>
              <CardDescription>Thème et apparence de l&apos;interface</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label>Thème</Label>
                <div className="grid grid-cols-3 gap-4">
                  <div className="p-4 border rounded-lg cursor-pointer hover:bg-muted">
                    <div className="h-20 bg-white border rounded mb-2"></div>
                    <div className="text-sm text-center">Clair</div>
                  </div>
                  <div className="p-4 border rounded-lg cursor-pointer hover:bg-muted bg-accent">
                    <div className="h-20 bg-gray-900 border rounded mb-2"></div>
                    <div className="text-sm text-center">Sombre</div>
                  </div>
                  <div className="p-4 border rounded-lg cursor-pointer hover:bg-muted">
                    <div className="h-20 bg-gradient-to-br from-white to-gray-900 border rounded mb-2"></div>
                    <div className="text-sm text-center">Auto</div>
                  </div>
                </div>
              </div>

              <div className="space-y-2">
                <Label>Couleur Primaire</Label>
                <div className="flex gap-2">
                  {['#10b981', '#3b82f6', '#8b5cf6', '#f59e0b', '#ec4899'].map((color) => (
                    <div
                      key={color}
                      className="w-10 h-10 rounded-full cursor-pointer border-2 border-border"
                      style={{ backgroundColor: color }}
                    />
                  ))}
                </div>
              </div>

              <div className="space-y-2">
                <Label>Densité</Label>
                <Select defaultValue="normal">
                  <option value="compact">Compacte</option>
                  <option value="normal">Normale</option>
                  <option value="comfortable">Confortable</option>
                </Select>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
